import { MapIcon, Menu, Settings, Search, MessageSquare, LogOut, Filter, Shield, Users } from 'lucide-react';
import { useState } from 'react';
import { ViewOptionsMenu } from './ViewOptionsMenu';
import { FilterMenu } from './FilterMenu';
import { LanguageSelector } from './LanguageSelector';
import { translations } from '../translations';

interface TopBarProps {
  mapType: 'standard' | 'satellite' | 'transit';
  setMapType: (type: 'standard' | 'satellite' | 'transit') => void;
  ecoMode: boolean;
  setEcoMode: (enabled: boolean) => void;
  isLoggedIn: boolean;
  username: string;
  onLoginClick: () => void;
  onLogoutClick: () => void;
  searchQuery: string;
  setSearchQuery: (query: string) => void;
  onSubmitEventClick: () => void;
  onFeedbackClick: () => void;
  onRulesClick: () => void;
  onCommunityGovernanceClick: () => void;
  selectedCategories: string[];
  setSelectedCategories: (categories: string[]) => void;
  selectedAgeFilters: ('new' | 'old')[];
  setSelectedAgeFilters: (filters: ('new' | 'old')[]) => void;
  selectedVerificationFilters: ('verified' | 'trusted' | 'unverified')[];
  setSelectedVerificationFilters: (filters: ('verified' | 'trusted' | 'unverified')[]) => void;
  language: 'nl' | 'en';
  setLanguage: (lang: 'nl' | 'en') => void;
}

export function TopBar({
  mapType,
  setMapType,
  ecoMode,
  setEcoMode,
  isLoggedIn,
  username,
  onLoginClick,
  onLogoutClick,
  searchQuery,
  setSearchQuery,
  onSubmitEventClick,
  onFeedbackClick,
  onRulesClick,
  onCommunityGovernanceClick,
  selectedCategories,
  setSelectedCategories,
  selectedAgeFilters,
  setSelectedAgeFilters,
  selectedVerificationFilters,
  setSelectedVerificationFilters,
  language,
  setLanguage
}: TopBarProps) {
  const [showViewOptions, setShowViewOptions] = useState(false);
  const [showFilterMenu, setShowFilterMenu] = useState(false);
  const [showLoginHover, setShowLoginHover] = useState(false);

  const t = translations[language].topBar;

  return (
    <div className="bg-white border-b border-gray-200 px-6 py-4 flex items-center justify-between">
      <div className="flex items-center gap-4">
        <div className="flex items-center gap-2">
          <MapIcon className="w-6 h-6 text-orange-500" />
          <h1 className="text-gray-900">{t.title}</h1>
        </div>
        
        {/* Search bar */}
        <div className="relative">
          <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
          <input
            type="text"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            placeholder={t.searchPlaceholder}
            className="pl-10 pr-4 py-2 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-orange-500 focus:border-transparent w-64"
          />
        </div>
      </div>

      <div className="flex items-center gap-4">
        {isLoggedIn && (
          <button
            onClick={onSubmitEventClick}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <MapIcon className="w-4 h-4" />
            {t.submitEvent}
          </button>
        )}

        {isLoggedIn && (
          <button
            onClick={onFeedbackClick}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <MessageSquare className="w-4 h-4" />
            {t.giveFeedback}
          </button>
        )}

        {isLoggedIn && (
          <button
            onClick={onRulesClick}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <Shield className="w-4 h-4" />
            {t.rules}
          </button>
        )}

        {isLoggedIn && (
          <button
            onClick={onCommunityGovernanceClick}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <Users className="w-4 h-4" />
            {t.communityGovernance}
          </button>
        )}

        <div className="relative">
          <button
            onClick={() => setShowFilterMenu(!showFilterMenu)}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <Filter className="w-4 h-4" />
            {t.filter}
          </button>
          {showFilterMenu && (
            <FilterMenu
              selectedCategories={selectedCategories}
              setSelectedCategories={setSelectedCategories}
              selectedAgeFilters={selectedAgeFilters}
              setSelectedAgeFilters={setSelectedAgeFilters}
              selectedVerificationFilters={selectedVerificationFilters}
              setSelectedVerificationFilters={setSelectedVerificationFilters}
              onClose={() => setShowFilterMenu(false)}
              language={language}
            />
          )}
        </div>

        <div className="relative">
          <button
            onClick={() => setShowViewOptions(!showViewOptions)}
            className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded hover:bg-gray-50"
          >
            <Settings className="w-4 h-4" />
            {t.viewOptions}
          </button>
          {showViewOptions && (
            <ViewOptionsMenu
              mapType={mapType}
              setMapType={setMapType}
              ecoMode={ecoMode}
              setEcoMode={setEcoMode}
              onClose={() => setShowViewOptions(false)}
              language={language}
            />
          )}
        </div>

        <LanguageSelector language={language} setLanguage={setLanguage} />

        <div className="relative">
          <button
            onMouseEnter={() => setShowLoginHover(true)}
            onMouseLeave={() => setShowLoginHover(false)}
            onClick={onLoginClick}
            className="flex items-center gap-2 px-4 py-2 bg-orange-500 text-white rounded hover:bg-orange-600"
          >
            <MapIcon className="w-4 h-4" />
            {isLoggedIn ? `${t.loggedInAs}: ${username}` : t.loginMastodon}
          </button>
          {showLoginHover && !isLoggedIn && (
            <div className="absolute top-full mt-2 right-0 bg-white border border-gray-200 rounded shadow-lg p-4 w-80 z-50">
              <p className="text-sm text-gray-700">
                {t.loginHoverText}
              </p>
            </div>
          )}
        </div>

        {isLoggedIn && (
          <button
            onClick={onLogoutClick}
            className="flex items-center gap-2 px-4 py-2 bg-red-500 text-white rounded hover:bg-red-600"
          >
            <LogOut className="w-4 h-4" />
            {t.logout}
          </button>
        )}
      </div>
    </div>
  );
}