import { useState } from 'react';
import { X, Calendar, MapPin, FileText, Tag, RotateCcw, ZoomIn, ZoomOut } from 'lucide-react';
import mapImage from 'figma:asset/4150da1c92f4f17a924a0886a8ae9ca277f334c5.png';

interface SubmitEventFormProps {
  onClose: () => void;
  onAddEvent: (eventData: { name: string; description: string; location: string; category: string; date: string; position: { x: number; y: number } }) => void;
}

export function SubmitEventForm({ onClose, onAddEvent }: SubmitEventFormProps) {
  const [eventName, setEventName] = useState('');
  const [description, setDescription] = useState('');
  const [location, setLocation] = useState('');
  const [category, setCategory] = useState('');
  const [date, setDate] = useState('');
  const [position, setPosition] = useState<{ x: number; y: number } | null>(null);
  const [scale, setScale] = useState(1);
  const [pan, setPan] = useState({ x: 0, y: 0 });
  const [isDragging, setIsDragging] = useState(false);
  const [dragStart, setDragStart] = useState({ x: 0, y: 0 });

  const handleMapClick = (e: React.MouseEvent<HTMLDivElement>) => {
    if (isDragging) return;
    
    const rect = e.currentTarget.getBoundingClientRect();
    const x = ((e.clientX - rect.left) / rect.width) * 100;
    const y = ((e.clientY - rect.top) / rect.height) * 100;
    setPosition({ x, y });
  };

  const handleMouseDown = (e: React.MouseEvent<HTMLDivElement>) => {
    setIsDragging(true);
    setDragStart({ x: e.clientX - pan.x, y: e.clientY - pan.y });
  };

  const handleMouseMove = (e: React.MouseEvent<HTMLDivElement>) => {
    if (!isDragging) return;
    setPan({
      x: e.clientX - dragStart.x,
      y: e.clientY - dragStart.y
    });
  };

  const handleMouseUp = () => {
    setIsDragging(false);
  };

  const handleReset = () => {
    setScale(1);
    setPan({ x: 0, y: 0 });
    setPosition(null);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!position) {
      alert('Klik op de kaart om een locatie te selecteren');
      return;
    }
    onAddEvent({ name: eventName, description, location, category, date, position });
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto p-6 relative">
        <button
          onClick={onClose}
          className="absolute top-4 right-4 p-2 hover:bg-gray-100 rounded-full z-10"
        >
          <X className="w-5 h-5" />
        </button>

        <div className="mb-6">
          <h2 className="text-gray-900 mb-2">Stuur nieuwe evenementen in naar de Gemeente Amsterdam</h2>
          <p className="text-sm text-gray-600">
            Vul onderstaande gegevens in om een nieuw evenement voor te stellen. De gemeente zal uw voorstel beoordelen.
          </p>
        </div>

        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <label htmlFor="eventName" className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
              <Calendar className="w-4 h-4" />
              Naam evenement
            </label>
            <input
              type="text"
              id="eventName"
              value={eventName}
              onChange={(e) => setEventName(e.target.value)}
              placeholder="Bijvoorbeeld: Zomerfestival Jordaan"
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500"
              required
            />
          </div>

          <div>
            <label htmlFor="description" className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
              <FileText className="w-4 h-4" />
              Beschrijving
            </label>
            <textarea
              id="description"
              value={description}
              onChange={(e) => setDescription(e.target.value)}
              placeholder="Geef een korte beschrijving van het evenement..."
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500 min-h-24"
              required
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label htmlFor="location" className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
                <MapPin className="w-4 h-4" />
                Locatie
              </label>
              <input
                type="text"
                id="location"
                value={location}
                onChange={(e) => setLocation(e.target.value)}
                placeholder="Adres of buurt"
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500"
                required
              />
            </div>

            <div>
              <label htmlFor="date" className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
                <Calendar className="w-4 h-4" />
                Datum
              </label>
              <input
                type="date"
                id="date"
                value={date}
                onChange={(e) => setDate(e.target.value)}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500"
                required
              />
            </div>
          </div>

          <div>
            <label htmlFor="category" className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
              <Tag className="w-4 h-4" />
              Categorie
            </label>
            <select
              id="category"
              value={category}
              onChange={(e) => setCategory(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-orange-500"
              required
            >
              <option value="">Selecteer een categorie</option>
              <option value="cultuur">Cultuur</option>
              <option value="sport">Sport</option>
              <option value="muziek">Muziek</option>
              <option value="food">Food & Drink</option>
              <option value="markt">Markt</option>
              <option value="kinderen">Kinderen</option>
              <option value="anders">Anders</option>
            </select>
          </div>

          {/* Map Location Selector */}
          <div>
            <label className="block text-sm text-gray-700 mb-1 flex items-center gap-2">
              <MapPin className="w-4 h-4" />
              Klik op de kaart om de locatie te selecteren
            </label>
            <div 
              className="relative w-full h-96 border-2 border-gray-300 rounded-lg overflow-hidden cursor-crosshair hover:border-orange-500 transition-colors"
              onClick={handleMapClick}
            >
              <div
                className="w-full h-full"
                onMouseDown={handleMouseDown}
                onMouseMove={handleMouseMove}
                onMouseUp={handleMouseUp}
                onMouseLeave={handleMouseUp}
                style={{
                  transform: `scale(${scale}) translate(${pan.x / scale}px, ${pan.y / scale}px)`,
                  transformOrigin: 'center center',
                  transition: isDragging ? 'none' : 'transform 0.1s ease-out',
                  cursor: isDragging ? 'grabbing' : 'crosshair'
                }}
              >
                <img
                  src={mapImage}
                  alt="Amsterdam kaart"
                  className="w-full h-full object-cover pointer-events-none"
                />
              </div>
              {position && (
                <div
                  className="absolute w-6 h-6 -ml-3 -mt-6 pointer-events-none"
                  style={{
                    left: `${position.x}%`,
                    top: `${position.y}%`
                  }}
                >
                  <MapPin className="w-6 h-6 text-orange-500 fill-orange-500 drop-shadow-lg" />
                </div>
              )}
            </div>
            {position ? (
              <p className="text-xs text-green-600 mt-1">
                ✓ Locatie geselecteerd op de kaart
              </p>
            ) : (
              <p className="text-xs text-gray-500 mt-1">
                Klik op de kaart om de positie van uw evenement aan te geven
              </p>
            )}
            
            {/* Zoom Controls */}
            <div className="flex items-center gap-3 mt-3">
              <button
                type="button"
                onClick={() => setScale(Math.max(1, scale - 0.2))}
                className="bg-gray-100 p-2 rounded-full hover:bg-gray-200 transition-colors"
                title="Uitzoomen"
              >
                <ZoomOut className="w-4 h-4 text-gray-700" />
              </button>
              
              <div className="flex-1 flex items-center gap-2">
                <span className="text-xs text-gray-600 min-w-[3rem]">Zoom</span>
                <input
                  type="range"
                  min="1"
                  max="3"
                  step="0.1"
                  value={scale}
                  onChange={(e) => setScale(parseFloat(e.target.value))}
                  className="flex-1 h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer accent-orange-500"
                />
                <span className="text-xs text-gray-600 min-w-[3rem] text-right">{Math.round(scale * 100)}%</span>
              </div>
              
              <button
                type="button"
                onClick={() => setScale(Math.min(3, scale + 0.2))}
                className="bg-gray-100 p-2 rounded-full hover:bg-gray-200 transition-colors"
                title="Inzoomen"
              >
                <ZoomIn className="w-4 h-4 text-gray-700" />
              </button>
              
              <button
                type="button"
                onClick={handleReset}
                className="bg-orange-100 p-2 rounded-full hover:bg-orange-200 transition-colors"
                title="Kaart resetten"
              >
                <RotateCcw className="w-4 h-4 text-orange-700" />
              </button>
            </div>
          </div>

          <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
            <p className="text-sm text-gray-700">
              ℹ️ Na indiening zal de Gemeente Amsterdam uw account verifiëren. Het kan een paar weken duren totdat je account geverifieerd is. U ontvangt een notificatie via Mastodon zodra het is goedgekeurd.
            </p>
          </div>

          <div className="bg-orange-50 border border-orange-200 rounded-lg p-3">
            <p className="text-sm text-gray-700">
              ⏱️ Het kan een paar weken duren totdat je evenement is geverifieerd door de gemeente Amsterdam
            </p>
          </div>

          <div className="flex gap-3 pt-4">
            <button
              type="button"
              onClick={onClose}
              className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
            >
              Annuleren
            </button>
            <button
              type="submit"
              className="flex-1 px-4 py-2 bg-orange-500 text-white rounded-lg hover:bg-orange-600"
            >
              Evenement indienen
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}