import { useState } from 'react';
import { EventMarker } from './EventMarker';
import { DiscussionMarker } from './DiscussionMarker';
import { TransitLines } from './TransitLines';
import { ZoomControl } from './ZoomControl';
import mapImage from 'figma:asset/e90d93f8b3a315dd22a6813c79daf2e1a0dbf17e.png';
import satelliteImage from 'figma:asset/8eaec69f14515647838e17574eeec5a0915d1d31.png';
import { discussionsData } from '../data/eventData';

interface MapViewProps {
  mapType: 'standard' | 'satellite' | 'transit';
  ecoMode: boolean;
  isLoggedIn: boolean;
  searchQuery: string;
  events: Event[];
  highlightedEventId: string | null;
  selectedCategories: string[];
  selectedAgeFilters: ('new' | 'old')[];
  selectedVerificationFilters: ('verified' | 'trusted' | 'unverified')[];
  language: 'nl' | 'en';
}

interface Event {
  id: string;
  name: string;
  description: string;
  age: 'new' | 'old';
  verification: 'verified' | 'trusted' | 'unverified';
  position: { x: number; y: number };
  category: string;
}

interface Discussion {
  id: string;
  neighborhood: string;
  hashtag: string;
  topic: string;
  position: { x: number; y: number };
}

const mockEvents: Event[] = [
  {
    id: '1',
    name: 'Straatfestival Centrum',
    description: 'Jaarlijks straatfestival in het centrum van Amsterdam',
    age: 'new',
    verification: 'verified',
    position: { x: 45, y: 30 },
    category: 'Festival'
  },
  {
    id: '2',
    name: 'Kunstmarkt Jordaan',
    description: 'Maandelijkse kunstmarkt in de Jordaan',
    age: 'old',
    verification: 'trusted',
    position: { x: 25, y: 45 },
    category: 'Markt'
  },
  {
    id: '3',
    name: 'Historische rondleiding',
    description: 'Rondleiding door historisch Amsterdam',
    age: 'old',
    verification: 'unverified',
    position: { x: 55, y: 35 },
    category: 'Cultuur'
  },
  {
    id: '4',
    name: 'Muziekfestival Oost',
    description: 'Zomers muziekfestival in Amsterdam Oost',
    age: 'new',
    verification: 'verified',
    position: { x: 70, y: 50 },
    category: 'Muziek'
  },
  {
    id: '5',
    name: 'Boekenmarkt',
    description: 'Tweedehands boekenmarkt bij het water',
    age: 'old',
    verification: 'trusted',
    position: { x: 60, y: 25 },
    category: 'Markt'
  },
  {
    id: '6',
    name: 'Filmavond park',
    description: 'Openlucht filmvoorstelling in het Vondelpark',
    age: 'new',
    verification: 'verified',
    position: { x: 20, y: 60 },
    category: 'Film'
  },
  {
    id: '7',
    name: 'Sportdag West',
    description: 'Gemeentelijke sportdag voor alle leeftijden',
    age: 'old',
    verification: 'verified',
    position: { x: 15, y: 35 },
    category: 'Sport'
  },
  {
    id: '8',
    name: 'Food festival Zuid',
    description: 'Internationale food trucks festival',
    age: 'new',
    verification: 'trusted',
    position: { x: 50, y: 70 },
    category: 'Food'
  }
];

export function MapView({ mapType, ecoMode, isLoggedIn, searchQuery, events, highlightedEventId, selectedCategories, selectedAgeFilters, selectedVerificationFilters, language }: MapViewProps) {
  const [hoveredEvent, setHoveredEvent] = useState<string | null>(null);
  const [allowedRemoteContent, setAllowedRemoteContent] = useState<Set<string>>(new Set());
  const [zoom, setZoom] = useState(0);
  const [pan, setPan] = useState({ x: 0, y: 0 });
  const [isDragging, setIsDragging] = useState(false);
  const [dragStart, setDragStart] = useState({ x: 0, y: 0 });

  // Apply search filter
  let filteredEvents = searchQuery
    ? events.filter(event =>
        event.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        event.description.toLowerCase().includes(searchQuery.toLowerCase())
      )
    : events;

  // Apply age filter
  if (selectedAgeFilters.length > 0 && selectedAgeFilters.length < 2) {
    filteredEvents = filteredEvents.filter(event => 
      selectedAgeFilters.includes(event.age)
    );
  }

  // Apply category filter
  if (selectedCategories.length > 0) {
    filteredEvents = filteredEvents.filter(event => 
      selectedCategories.includes(event.category)
    );
  }

  // Apply verification filter
  if (selectedVerificationFilters.length > 0) {
    filteredEvents = filteredEvents.filter(event => 
      selectedVerificationFilters.includes(event.verification)
    );
  }

  const handleLoadRemoteContent = (eventId: string) => {
    setAllowedRemoteContent(prev => new Set([...prev, eventId]));
  };

  const handleMouseDown = (e: React.MouseEvent) => {
    if (e.button === 0) {
      setIsDragging(true);
      setDragStart({ x: e.clientX, y: e.clientY });
    }
  };

  const handleMouseMove = (e: React.MouseEvent) => {
    if (isDragging) {
      const deltaX = e.clientX - dragStart.x;
      const deltaY = e.clientY - dragStart.y;
      
      setPan(prev => ({
        x: prev.x + deltaX,
        y: prev.y + deltaY
      }));
      
      setDragStart({ x: e.clientX, y: e.clientY });
    }
  };

  const handleMouseUp = () => {
    setIsDragging(false);
  };

  const scale = 1 + (zoom / 100);

  return (
    <div className="w-full h-full relative overflow-hidden bg-gray-100">
      {/* Map container with zoom and pan */}
      <div
        className="w-full h-full relative"
        style={{
          transform: `scale(${scale}) translate(${pan.x / scale}px, ${pan.y / scale}px)`,
          transformOrigin: 'center center',
          transition: isDragging ? 'none' : 'transform 0.1s ease-out',
          cursor: isDragging ? 'grabbing' : 'grab'
        }}
        onMouseDown={handleMouseDown}
        onMouseMove={handleMouseMove}
        onMouseUp={handleMouseUp}
        onMouseLeave={handleMouseUp}
      >
        {/* Map Background - gebruik altijd de eerste foto */}
        <div className="w-full h-full">
          <img
            src={mapType === 'satellite' ? satelliteImage : mapImage}
            alt="Amsterdam kaart"
            className="w-full h-full object-cover pointer-events-none"
          />
        </div>
        
        {/* Transit Lines overlay voor OV weergave */}
        {mapType === 'transit' && <TransitLines />}

        {/* Event Markers */}
        {filteredEvents.map(event => (
          <EventMarker
            key={event.id}
            event={event}
            isHovered={hoveredEvent === event.id}
            onHover={setHoveredEvent}
            ecoMode={ecoMode}
            allowedRemoteContent={allowedRemoteContent.has(event.id)}
            onLoadRemoteContent={handleLoadRemoteContent}
            isLoggedIn={isLoggedIn}
            searchQuery={searchQuery}
            isHighlighted={highlightedEventId === event.id}
            language={language}
          />
        ))}

        {/* Discussion Markers - only visible when logged in */}
        {isLoggedIn && discussionsData.map(discussion => (
          <DiscussionMarker
            key={discussion.id}
            discussion={discussion}
            isLoggedIn={isLoggedIn}
            ecoMode={ecoMode}
            allowedRemoteContent={allowedRemoteContent.has(discussion.id)}
            onLoadRemoteContent={handleLoadRemoteContent}
            language={language}
          />
        ))}
      </div>
      
      {/* Zoom Control */}
      <ZoomControl zoom={zoom} onZoomChange={setZoom} onResetView={() => { setZoom(0); setPan({ x: 0, y: 0 }); }} />
    </div>
  );
}