import { useState } from 'react';
import { ChevronDown, ChevronUp, Search, Menu, MapIcon, LogOut } from 'lucide-react';
import { Legend } from './Legend';
import { translations } from '../translations';

interface InfoPanelProps {
  mapType: 'standard' | 'satellite' | 'transit';
  onMoreFunctionsClick: () => void;
  isLoggedIn: boolean;
  username: string;
  onLoginClick: () => void;
  onLogoutClick: () => void;
  language: 'nl' | 'en';
}

export function InfoPanel({ mapType, onMoreFunctionsClick, isLoggedIn, username, onLoginClick, onLogoutClick, language }: InfoPanelProps) {
  const [showLegend, setShowLegend] = useState(false);

  const t = translations[language].infoPanel;

  return (
    <div className="absolute top-4 right-4 bg-white border border-gray-200 rounded-lg shadow-lg w-80 z-10">
      <div className="p-4">
        <h2 className="text-gray-900 mb-2">
          {t.welcomeTitle}
        </h2>
        
        <p className="text-sm text-gray-600 mb-2">
          {t.seeEvents}
        </p>
        
        <p className="text-sm text-gray-700 mb-2">
          {t.submitIdeas}
        </p>

        <p className="text-sm text-gray-700 mb-2">
          {t.participate}
        </p>

        <p className="text-sm text-gray-700 mb-4">
          {t.getResponse}
        </p>

        <div className="p-3 bg-blue-50 border border-blue-200 rounded-lg mb-4">
          <p className="text-sm text-gray-700">
            {t.moreFeaturesInfo}
          </p>
        </div>

        {/* Mastodon Login Button - Prominent */}
        {!isLoggedIn && (
          <div className="mb-4 p-3 bg-orange-50 border-2 border-orange-500 rounded-lg">
            <button
              onClick={onLoginClick}
              className="w-full flex items-center justify-center gap-2 px-4 py-3 bg-orange-500 text-white rounded-lg hover:bg-orange-600 transition-colors"
            >
              <MapIcon className="w-5 h-5" />
              <span>{translations[language].topBar.loginMastodon}</span>
            </button>
            <p className="text-xs text-gray-600 text-center mt-2">
              {t.loginRequired}
            </p>
          </div>
        )}

        {isLoggedIn && (
          <div className="mb-4 p-3 bg-green-50 border-2 border-green-500 rounded-lg">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                <span className="text-sm text-gray-900">{t.loggedInAs}: {username}</span>
              </div>
              <button
                onClick={onLogoutClick}
                className="text-sm text-red-600 hover:text-red-700"
              >
                <LogOut className="w-4 h-4" />
              </button>
            </div>
          </div>
        )}

        <button
          onClick={() => setShowLegend(!showLegend)}
          className="flex items-center gap-2 text-sm text-gray-700 hover:text-gray-900"
        >
          {showLegend ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
          {t.showLegend}
        </button>

        {showLegend && <Legend mapType={mapType} />}

        <button
          onClick={onMoreFunctionsClick}
          className="flex items-center gap-2 text-sm text-gray-700 hover:text-gray-900 mt-3"
        >
          <Menu className="w-4 h-4" />
          {t.moreFunctions}
        </button>
      </div>
    </div>
  );
}