import { X, Filter, Verified, AlertCircle, ShieldAlert, Calendar } from 'lucide-react';
import { useEffect, useRef } from 'react';
import { translations } from '../translations';

interface FilterMenuProps {
  selectedCategories: string[];
  setSelectedCategories: (categories: string[]) => void;
  selectedAgeFilters: ('new' | 'old')[];
  setSelectedAgeFilters: (filters: ('new' | 'old')[]) => void;
  selectedVerificationFilters: ('verified' | 'trusted' | 'unverified')[];
  setSelectedVerificationFilters: (filters: ('verified' | 'trusted' | 'unverified')[]) => void;
  onClose: () => void;
  language: 'nl' | 'en';
}

const CATEGORIES = [
  'Festival',
  'Markt',
  'Cultuur',
  'Muziek',
  'Film',
  'Sport',
  'Food',
  'Theater',
  'Gemeente',
  'Workshop',
  'Sociaal',
  'Natuur'
];

export function FilterMenu({
  selectedCategories,
  setSelectedCategories,
  selectedAgeFilters,
  setSelectedAgeFilters,
  selectedVerificationFilters,
  setSelectedVerificationFilters,
  onClose,
  language
}: FilterMenuProps) {
  const menuRef = useRef<HTMLDivElement>(null);
  const t = translations[language].filterMenu;
  const tLegend = translations[language].legend;

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (menuRef.current && !menuRef.current.contains(event.target as Node)) {
        onClose();
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, [onClose]);

  const toggleCategory = (category: string) => {
    if (selectedCategories.includes(category)) {
      setSelectedCategories(selectedCategories.filter(c => c !== category));
    } else {
      setSelectedCategories([...selectedCategories, category]);
    }
  };

  const toggleAgeFilter = (filter: 'new' | 'old') => {
    if (selectedAgeFilters.includes(filter)) {
      setSelectedAgeFilters(selectedAgeFilters.filter(f => f !== filter));
    } else {
      setSelectedAgeFilters([...selectedAgeFilters, filter]);
    }
  };

  const toggleVerificationFilter = (filter: 'verified' | 'trusted' | 'unverified') => {
    if (selectedVerificationFilters.includes(filter)) {
      setSelectedVerificationFilters(selectedVerificationFilters.filter(f => f !== filter));
    } else {
      setSelectedVerificationFilters([...selectedVerificationFilters, filter]);
    }
  };

  const handleSelectAllCategories = () => {
    setSelectedCategories(CATEGORIES);
  };

  const handleDeselectAllCategories = () => {
    setSelectedCategories([]);
  };

  const handleResetFilters = () => {
    setSelectedCategories([]);
    setSelectedAgeFilters(['new', 'old']);
    setSelectedVerificationFilters(['verified', 'trusted', 'unverified']);
  };

  return (
    <div
      ref={menuRef}
      className="absolute top-full mt-2 right-0 bg-white border border-gray-200 rounded-lg shadow-xl w-80 z-50"
    >
      <div className="p-4">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center gap-2">
            <Filter className="w-5 h-5 text-gray-700" />
            <h3 className="text-gray-900">{t.title}</h3>
          </div>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        {/* Age Filters */}
        <div className="mb-4 pb-4 border-b border-gray-200">
          <h4 className="text-sm text-gray-700 mb-2">{t.age}</h4>
          <div className="space-y-2">
            <label className="flex items-center gap-2 cursor-pointer">
              <input
                type="checkbox"
                checked={selectedAgeFilters.includes('new')}
                onChange={() => toggleAgeFilter('new')}
                className="w-4 h-4 text-green-600 rounded focus:ring-2 focus:ring-green-500"
              />
              <div className="flex items-center gap-2">
                <div className="w-4 h-4 rounded-full bg-green-500 border-2 border-white shadow" />
                <span className="text-sm text-gray-700">{t.newEvent}</span>
              </div>
            </label>
            <label className="flex items-center gap-2 cursor-pointer">
              <input
                type="checkbox"
                checked={selectedAgeFilters.includes('old')}
                onChange={() => toggleAgeFilter('old')}
                className="w-4 h-4 text-red-600 rounded focus:ring-2 focus:ring-red-500"
              />
              <div className="flex items-center gap-2">
                <div className="w-4 h-4 rounded-full bg-red-500 border-2 border-white shadow" />
                <span className="text-sm text-gray-700">{t.oldEvent}</span>
              </div>
            </label>
          </div>
        </div>

        {/* Verification Filters */}
        <div className="mb-4 pb-4 border-b border-gray-200">
          <h4 className="text-sm text-gray-700 mb-2">{t.verification}</h4>
          <div className="space-y-2">
            <label className="flex items-center gap-2 cursor-pointer">
              <input
                type="checkbox"
                checked={selectedVerificationFilters.includes('verified')}
                onChange={() => toggleVerificationFilter('verified')}
                className="w-4 h-4 text-green-600 rounded focus:ring-2 focus:ring-green-500"
              />
              <div className="flex items-center gap-2">
                <Verified className="w-4 h-4 text-green-500" />
                <span className="text-sm text-gray-700">{tLegend.verified}</span>
              </div>
            </label>
            <label className="flex items-center gap-2 cursor-pointer">
              <input
                type="checkbox"
                checked={selectedVerificationFilters.includes('trusted')}
                onChange={() => toggleVerificationFilter('trusted')}
                className="w-4 h-4 text-orange-600 rounded focus:ring-2 focus:ring-orange-500"
              />
              <div className="flex items-center gap-2">
                <AlertCircle className="w-4 h-4 text-orange-500" />
                <span className="text-sm text-gray-700">{tLegend.trusted}</span>
              </div>
            </label>
            <label className="flex items-center gap-2 cursor-pointer">
              <input
                type="checkbox"
                checked={selectedVerificationFilters.includes('unverified')}
                onChange={() => toggleVerificationFilter('unverified')}
                className="w-4 h-4 text-red-600 rounded focus:ring-2 focus:ring-red-500"
              />
              <div className="flex items-center gap-2">
                <ShieldAlert className="w-4 h-4 text-red-500" />
                <span className="text-sm text-gray-700">{tLegend.unverified}</span>
              </div>
            </label>
          </div>
        </div>

        {/* Category Filters */}
        <div className="mb-4">
          <div className="flex items-center justify-between mb-2">
            <h4 className="text-sm text-gray-700">{t.categories}</h4>
            <div className="flex gap-2">
              <button
                onClick={handleSelectAllCategories}
                className="text-xs text-blue-600 hover:text-blue-700"
              >
                {t.selectAll}
              </button>
              <span className="text-xs text-gray-400">|</span>
              <button
                onClick={handleDeselectAllCategories}
                className="text-xs text-blue-600 hover:text-blue-700"
              >
                {t.deselectAll}
              </button>
            </div>
          </div>
          <div className="max-h-64 overflow-y-auto space-y-2">
            {CATEGORIES.map(category => (
              <label key={category} className="flex items-center gap-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={selectedCategories.includes(category)}
                  onChange={() => toggleCategory(category)}
                  className="w-4 h-4 text-orange-600 rounded focus:ring-2 focus:ring-orange-500"
                />
                <span className="text-sm text-gray-700">{category}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Reset Button */}
        <button
          onClick={handleResetFilters}
          className="w-full px-4 py-2 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm"
        >
          {t.apply}
        </button>
      </div>
    </div>
  );
}