import { useState } from 'react';
import { X, Send, MessageSquare } from 'lucide-react';

interface FeedbackModalProps {
  onClose: () => void;
  onSubmit: (feedback: { name: string; email: string; category: string; message: string }) => void;
}

export function FeedbackModal({ onClose, onSubmit }: FeedbackModalProps) {
  const [name, setName] = useState('');
  const [email, setEmail] = useState('');
  const [category, setCategory] = useState('algemeen');
  const [message, setMessage] = useState('');

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (message.trim()) {
      onSubmit({ name, email, category, message });
      onClose();
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-md p-6">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center gap-2">
            <MessageSquare className="w-5 h-5 text-orange-500" />
            <h2 className="text-gray-900">Geef feedback</h2>
          </div>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        <p className="text-sm text-gray-600 mb-4">
          Deel uw feedback over uw gebruikerservaring of geef verbetersuggesties voor de gemeenschapsbrievenbus met de Gemeente Amsterdam.
        </p>

        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <label className="block text-sm text-gray-700 mb-1">
              Naam (optioneel)
            </label>
            <input
              type="text"
              value={name}
              onChange={(e) => setName(e.target.value)}
              placeholder="Uw naam"
              className="w-full px-3 py-2 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-orange-500"
            />
          </div>

          <div>
            <label className="block text-sm text-gray-700 mb-1">
              E-mail (optioneel)
            </label>
            <input
              type="email"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              placeholder="uw.email@example.com"
              className="w-full px-3 py-2 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-orange-500"
            />
          </div>

          <div>
            <label className="block text-sm text-gray-700 mb-1">
              Categorie
            </label>
            <select
              value={category}
              onChange={(e) => setCategory(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-orange-500"
            >
              <option value="algemeen">Algemeen</option>
              <option value="technisch">Technische problemen</option>
              <option value="suggestie">Suggestie voor verbetering</option>
              <option value="event">Over evenementen</option>
              <option value="wijkdiscussie">Over wijkdiscussies</option>
              <option value="andere">Andere</option>
            </select>
          </div>

          <div>
            <label className="block text-sm text-gray-700 mb-1">
              Uw feedback *
            </label>
            <textarea
              value={message}
              onChange={(e) => setMessage(e.target.value)}
              placeholder="Vertel ons wat u denkt over dit platform..."
              required
              rows={5}
              className="w-full px-3 py-2 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-orange-500 resize-none"
            />
          </div>

          <div className="flex gap-3">
            <button
              type="button"
              onClick={onClose}
              className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded hover:bg-gray-50"
            >
              Annuleren
            </button>
            <button
              type="submit"
              className="flex-1 px-4 py-2 bg-orange-500 text-white rounded hover:bg-orange-600 flex items-center justify-center gap-2"
            >
              <Send className="w-4 h-4" />
              Verstuur feedback
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}