import { useState } from 'react';
import { Verified, AlertCircle, ShieldAlert, Calendar, MessageSquare, Send, Star } from 'lucide-react';
import { translations } from '../translations';
import { eventReviewsData, ReviewData } from '../data/reviewsData';

interface Event {
  id: string;
  name: string;
  description: string;
  age: 'new' | 'old';
  verification: 'verified' | 'trusted' | 'unverified';
  position: { x: number; y: number };
  category: string;
  date: string;
}

interface EventMarkerProps {
  event: Event;
  isHovered: boolean;
  onHover: (id: string | null) => void;
  ecoMode: boolean;
  allowedRemoteContent: boolean;
  onLoadRemoteContent: (id: string) => void;
  isLoggedIn: boolean;
  searchQuery: string;
  isHighlighted: boolean;
  language: 'nl' | 'en';
}

interface Review {
  id: string;
  author: string;
  avatar: string;
  content: string;
  rating: number;
  timestamp: string;
}

// Helper function to convert ReviewData to Review based on language
const convertToReview = (reviewData: ReviewData, language: 'nl' | 'en'): Review => ({
  id: reviewData.id,
  author: reviewData.author[language],
  avatar: reviewData.avatar,
  content: reviewData.content[language],
  rating: reviewData.rating,
  timestamp: reviewData.timestamp[language]
});

export function EventMarker({
  event,
  isHovered,
  onHover,
  ecoMode,
  allowedRemoteContent,
  onLoadRemoteContent,
  isLoggedIn,
  searchQuery,
  isHighlighted,
  language
}: EventMarkerProps) {
  const [isExpanded, setIsExpanded] = useState(false);
  const [reviews, setReviews] = useState<Review[]>(eventReviewsData[event.id]?.map(reviewData => convertToReview(reviewData, language)) || []);
  const [newReview, setNewReview] = useState('');
  const [newRating, setNewRating] = useState(5);

  const t = translations[language].eventMarker;

  const getAgeColor = () => {
    switch (event.age) {
      case 'new':
        return 'bg-green-500';
      case 'old':
        return 'bg-red-500 opacity-60';
    }
  };

  const getMarkerSize = () => {
    switch (event.age) {
      case 'new':
        return 'w-5 h-5'; // Groter voor nieuwe events
      case 'old':
        return 'w-3 h-3'; // Kleiner voor oude events
    }
  };

  const getIconSize = () => {
    switch (event.age) {
      case 'new':
        return 'w-2.5 h-2.5';
      case 'old':
        return 'w-1.5 h-1.5';
    }
  };

  const getVerificationIcon = () => {
    switch (event.verification) {
      case 'verified':
        return <Verified className="w-4 h-4 text-green-500" />;
      case 'trusted':
        return <AlertCircle className="w-4 h-4 text-orange-500" />;
      case 'unverified':
        return <ShieldAlert className="w-4 h-4 text-red-500" />;
    }
  };

  const getVerificationText = () => {
    switch (event.verification) {
      case 'verified':
        return 'Geverifieerd door Gemeente Amsterdam';
      case 'trusted':
        return 'Betrouwbaar, maar niet geverifieerd door de gemeente Amsterdam';
      case 'unverified':
        return 'Niet geverifieerd';
    }
  };

  const handleSubmitReview = () => {
    if (newReview.trim()) {
      const newReviewObj: Review = {
        id: `r${Date.now()}`,
        author: 'Jouw naam',
        avatar: '👤',
        content: newReview.trim(),
        rating: newRating,
        timestamp: 'Zojuist'
      };
      
      setReviews([...reviews, newReviewObj]);
      setNewReview('');
      setNewRating(5);
    }
  };

  const showInfo = isHovered && (!ecoMode || allowedRemoteContent) && !isExpanded;

  // Check if this event matches the search query
  const isSearchMatch = searchQuery && (
    event.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    event.description.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div
      className={`absolute transform -translate-x-1/2 -translate-y-1/2 ${isExpanded ? 'z-[100]' : 'z-20'}`}
      style={{ left: `${event.position.x}%`, top: `${event.position.y}%` }}
      onMouseEnter={() => !isExpanded && onHover(event.id)}
      onMouseLeave={() => !isExpanded && onHover(null)}
    >
      <div className="flex items-center gap-2">
        {/* Marker dot - hide when info card or expanded view is shown */}
        {!showInfo && !isExpanded && (
          <div className="relative">
            {/* Blinking ring effect */}
            {isHighlighted && (
              <div 
                className={`absolute inset-0 ${getMarkerSize()} rounded-full ring-8 ring-yellow-400`}
                style={{ 
                  animation: 'blink-ring 0.6s ease-in-out infinite',
                  left: 0,
                  top: 0
                }}
              />
            )}
            <div
              className={`${getMarkerSize()} rounded-full ${getAgeColor()} ${
                isSearchMatch ? 'ring-4 ring-red-500' : 'border-2 border-white'
              } shadow-lg cursor-pointer hover:scale-110 transition-transform flex items-center justify-center relative ${
                event.age === 'new' ? 'z-30' : 'z-20'
              }`}
              style={isHighlighted ? { animation: 'blink 0.6s ease-in-out infinite' } : {}}
            >
              <Calendar className={`${getIconSize()} text-white`} />
            </div>
          </div>
        )}

        {/* Info card */}
        {showInfo && (
          <div className="bg-white rounded-lg shadow-xl p-4 w-64 border border-gray-200">
            <div className="flex items-start justify-between gap-2 mb-2">
              <h3 className="text-gray-900">{event.name}</h3>
              {getVerificationIcon()}
            </div>
            
            <p className="text-sm text-gray-600 mb-2">{event.description}</p>
            
            <div className="flex items-center gap-1 text-xs text-gray-700 mb-2">
              <Calendar className="w-3 h-3" />
              <span>{event.date}</span>
            </div>
            
            <div className="flex items-center gap-2 text-xs text-gray-500">
              {getVerificationIcon()}
              <span>{getVerificationText()}</span>
            </div>

            <div className="mt-2 text-xs text-gray-500">
              Categorie: {event.category}
            </div>

            <button
              onClick={() => setIsExpanded(true)}
              className="w-full mt-3 px-3 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 text-sm flex items-center justify-center gap-2"
            >
              <MessageSquare className="w-4 h-4" />
              {t.viewReviews}
            </button>
          </div>
        )}

        {/* Expanded reviews section */}
        {isExpanded && (
          <>
            {/* Backdrop overlay */}
            <div 
              className="fixed inset-0 bg-black bg-opacity-30 z-[90]"
              onClick={() => setIsExpanded(false)}
            />
            
            <div className="bg-white rounded-lg shadow-xl p-4 w-96 border border-gray-200 max-h-[600px] overflow-y-auto fixed left-1/2 top-1/2 transform -translate-x-1/2 -translate-y-1/2 z-[100]">
              <div className="flex items-start justify-between gap-2 mb-2">
                <div>
                  <h3 className="text-gray-900 mb-1">{event.name}</h3>
                  <p className="text-sm text-gray-600 mb-2">{event.description}</p>
                </div>
                {getVerificationIcon()}
              </div>

              <button
                onClick={() => setIsExpanded(false)}
                className="text-sm text-gray-500 hover:text-gray-700 mb-4"
              >
                {t.back}
              </button>

              {/* Reviews section */}
              <div className="space-y-4 mb-4 border-t border-gray-200 pt-4">
                <h4 className="text-sm text-gray-700">{t.reviews} ({reviews.length})</h4>
                
                {isLoggedIn ? (
                  reviews.map(review => (
                    <div key={review.id} className="bg-gray-50 rounded-lg p-3">
                      <div className="flex items-start gap-2 mb-2">
                        <span className="text-2xl">{review.avatar}</span>
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="text-sm text-gray-900">{review.author}</span>
                            <div className="flex items-center gap-0.5">
                              {[...Array(5)].map((_, i) => (
                                <Star
                                  key={i}
                                  className={`w-3 h-3 ${
                                    i < review.rating
                                      ? 'text-yellow-500 fill-yellow-500'
                                      : 'text-gray-300'
                                  }`}
                                />
                              ))}
                            </div>
                          </div>
                          <span className="text-xs text-gray-500">{review.timestamp}</span>
                          <p className="text-sm text-gray-700 mt-1">{review.content}</p>
                        </div>
                      </div>
                    </div>
                  ))
                ) : (
                  <div className="bg-gray-50 rounded-lg p-4 text-center">
                    <MessageSquare className="w-8 h-8 text-gray-400 mx-auto mb-2" />
                    <p className="text-sm text-gray-600">
                      {t.loginToSeeReviews}
                    </p>
                  </div>
                )}
              </div>

              {/* New review input - only if logged in */}
              {isLoggedIn && (
                <div className="border-t border-gray-200 pt-4">
                  <h4 className="text-sm text-gray-700 mb-2">{t.writeReview}</h4>
                  
                  {/* Rating selector */}
                  <div className="flex items-center gap-2 mb-2">
                    <span className="text-sm text-gray-600">{t.rating}:</span>
                    <div className="flex items-center gap-1">
                      {[1, 2, 3, 4, 5].map((rating) => (
                        <button
                          key={rating}
                          onClick={() => setNewRating(rating)}
                          className="focus:outline-none"
                        >
                          <Star
                            className={`w-5 h-5 ${
                              rating <= newRating
                                ? 'text-yellow-500 fill-yellow-500'
                                : 'text-gray-300'
                            } hover:text-yellow-400 transition-colors`}
                          />
                        </button>
                      ))}
                    </div>
                  </div>

                  <div className="flex flex-col gap-2">
                    <textarea
                      value={newReview}
                      onChange={(e) => setNewReview(e.target.value)}
                      placeholder={t.yourReview}
                      className="w-full px-3 py-2 border border-gray-300 rounded text-sm focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                      rows={3}
                    />
                    <button
                      onClick={handleSubmitReview}
                      className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 flex items-center justify-center gap-2"
                    >
                      <Send className="w-4 h-4" />
                      {t.submit}
                    </button>
                  </div>
                </div>
              )}

              {!isLoggedIn && (
                <div className="border-t border-gray-200 pt-4">
                  <p className="text-sm text-gray-500 text-center">
                    {t.loginToSeeReviews}
                  </p>
                </div>
              )}
            </div>
          </>
        )}

        {/* Eco mode - show button to load content */}
        {isHovered && ecoMode && !allowedRemoteContent && !isExpanded && (
          <div className="bg-white rounded-lg shadow-xl p-4 w-64 border border-gray-200">
            <p className="text-sm text-gray-700 mb-3">
              Remote content beschikbaar
            </p>
            <button
              onClick={() => onLoadRemoteContent(event.id)}
              className="w-full px-3 py-2 bg-orange-500 text-white rounded hover:bg-orange-600 text-sm"
            >
              Load remote content
            </button>
          </div>
        )}
      </div>
    </div>
  );
}