import { useState } from 'react';
import { Users, MessageSquare, Send } from 'lucide-react';
import { DiscussionData } from '../data/eventData';
import { translations } from '../translations';
import { discussionCommentsData, CommentData } from '../data/commentsData';

interface DiscussionMarkerProps {
  discussion: DiscussionData;
  isLoggedIn: boolean;
  ecoMode: boolean;
  allowedRemoteContent: boolean;
  onLoadRemoteContent: (id: string) => void;
  language: 'nl' | 'en';
}

interface Comment {
  id: string;
  author: string;
  avatar: string;
  content: string;
  timestamp: string;
}

// Helper function to convert CommentData to Comment based on language
const convertToComment = (commentData: CommentData, language: 'nl' | 'en'): Comment => ({
  id: commentData.id,
  author: commentData.author[language],
  avatar: commentData.avatar,
  content: commentData.content[language],
  timestamp: commentData.timestamp[language]
});

export function DiscussionMarker({ discussion, isLoggedIn, ecoMode, allowedRemoteContent, onLoadRemoteContent, language }: DiscussionMarkerProps) {
  const [isHovered, setIsHovered] = useState(false);
  const [isExpanded, setIsExpanded] = useState(false);
  const [newComment, setNewComment] = useState('');
  
  // Convert comments to the current language
  const commentsData = discussionCommentsData[discussion.id] || [];
  const [comments, setComments] = useState<Comment[]>(commentsData.map(c => convertToComment(c, language)));

  const t = translations[language].discussionMarker;

  // Update comments when language changes
  useState(() => {
    setComments(commentsData.map(c => convertToComment(c, language)));
  });

  const handleSubmitComment = () => {
    if (newComment.trim()) {
      const newCommentObj: Comment = {
        id: `c${Date.now()}`,
        author: 'Jouw naam',
        avatar: '👤',
        content: newComment.trim(),
        timestamp: t.justNow
      };
      
      setComments([...comments, newCommentObj]);
      setNewComment('');
    }
  };

  const showInfoCard = isHovered && (!ecoMode || allowedRemoteContent) && !isExpanded;
  const showEcoCard = isHovered && ecoMode && !allowedRemoteContent && !isExpanded;

  return (
    <div
      className={`absolute transform -translate-x-1/2 -translate-y-1/2 ${isExpanded ? 'z-[100]' : 'z-20'}`}
      style={{ left: `${discussion.position.x}%`, top: `${discussion.position.y}%` }}
      onMouseEnter={() => !isExpanded && setIsHovered(true)}
      onMouseLeave={() => !isExpanded && setIsHovered(false)}
    >
      <div className="flex items-start gap-2">
        {/* Discussion marker - hide when any info card is shown or when expanded */}
        {!showInfoCard && !showEcoCard && !isExpanded && (
          <div className="bg-white rounded-lg shadow-lg p-3 cursor-pointer hover:shadow-xl transition-shadow border-2 border-purple-500">
            <Users className="w-5 h-5 text-purple-600" />
          </div>
        )}

        {/* Info card on hover */}
        {showInfoCard && (
          <div className="bg-white rounded-lg shadow-xl p-4 w-72 border border-gray-200">
            <div className="flex items-start gap-2 mb-2">
              <MessageSquare className="w-5 h-5 text-purple-600 flex-shrink-0 mt-0.5" />
              <div>
                <h3 className="text-gray-900 mb-1">{discussion.neighborhood[language]}</h3>
                <p className="text-sm text-gray-600 mb-2">{discussion.topic[language]}</p>
                <span className="text-sm text-purple-600">{discussion.hashtag}</span>
              </div>
            </div>

            {isLoggedIn && (
              <button
                onClick={() => setIsExpanded(true)}
                className="w-full mt-3 px-3 py-2 bg-purple-600 text-white rounded hover:bg-purple-700 text-sm"
              >
                {language === 'nl' ? 'Discussieer mee op Mastodon' : 'Join discussion on Mastodon'}
              </button>
            )}
          </div>
        )}

        {/* Eco mode - show button to load discussion content */}
        {showEcoCard && (
          <div className="bg-white rounded-lg shadow-xl p-4 w-72 border border-gray-200">
            <div className="flex items-center gap-2 mb-3">
              <MessageSquare className="w-5 h-5 text-purple-600" />
              <p className="text-sm text-gray-700">
                Eco-modus actief
              </p>
            </div>
            <p className="text-sm text-gray-600 mb-3">
              Deze wijkdiscussie wordt niet automatisch geladen om energie te besparen.
            </p>
            <button
              onClick={() => onLoadRemoteContent(discussion.id)}
              className="w-full px-3 py-2 bg-green-600 text-white rounded hover:bg-green-700 text-sm"
            >
              Laad discussie-inhoud
            </button>
          </div>
        )}

        {/* Expanded discussion thread */}
        {isExpanded && (
          <>
            {/* Backdrop overlay */}
            <div 
              className="fixed inset-0 bg-black bg-opacity-30 z-[90]"
              onClick={() => setIsExpanded(false)}
            />
            
            <div className="bg-white rounded-lg shadow-xl p-4 w-96 border border-gray-200 max-h-[600px] overflow-y-auto fixed left-1/2 top-1/2 transform -translate-x-1/2 -translate-y-1/2 z-[100]">
              <div className="flex items-start gap-2 mb-4">
                <MessageSquare className="w-5 h-5 text-purple-600 flex-shrink-0 mt-0.5" />
                <div>
                  <h3 className="text-gray-900 mb-1">{discussion.neighborhood[language]}</h3>
                  <p className="text-sm text-gray-600 mb-2">{discussion.topic[language]}</p>
                  <span className="text-sm text-purple-600">{discussion.hashtag}</span>
                </div>
              </div>

              <button
                onClick={() => setIsExpanded(false)}
                className="text-sm text-gray-500 hover:text-gray-700 mb-4"
              >
                {language === 'nl' ? '← Terug' : '← Back'}
              </button>

              {/* Comments section */}
              <div className="space-y-4 mb-4 border-t border-gray-200 pt-4">
                <h4 className="text-sm text-gray-700">{t.comments} ({comments.length})</h4>
                
                {isLoggedIn ? (
                  comments.map(comment => (
                    <div key={comment.id} className="bg-gray-50 rounded-lg p-3">
                      <div className="flex items-start gap-2 mb-2">
                        <span className="text-2xl">{comment.avatar}</span>
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="text-sm text-gray-900">{comment.author}</span>
                            <span className="text-xs text-gray-500">{comment.timestamp}</span>
                          </div>
                          <p className="text-sm text-gray-700">{comment.content}</p>
                        </div>
                      </div>
                    </div>
                  ))
                ) : (
                  <div className="bg-gray-50 rounded-lg p-4 text-center">
                    <MessageSquare className="w-8 h-8 text-gray-400 mx-auto mb-2" />
                    <p className="text-sm text-gray-600">
                      {t.loginToSeeComments}
                    </p>
                  </div>
                )}
              </div>

              {/* New comment input */}
              {isLoggedIn && (
                <div className="border-t border-gray-200 pt-4">
                  <div className="flex gap-2">
                    <input
                      type="text"
                      value={newComment}
                      onChange={(e) => setNewComment(e.target.value)}
                      onKeyPress={(e) => e.key === 'Enter' && handleSubmitComment()}
                      placeholder={t.yourComment}
                      className="flex-1 px-3 py-2 border border-gray-300 rounded text-sm focus:outline-none focus:ring-2 focus:ring-purple-500"
                    />
                    <button
                      onClick={handleSubmitComment}
                      className="px-4 py-2 bg-purple-600 text-white rounded hover:bg-purple-700 flex items-center gap-2"
                    >
                      <Send className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              )}

              {!isLoggedIn && (
                <div className="border-t border-gray-200 pt-4">
                  <p className="text-sm text-gray-500 text-center">
                    {t.loginToSeeComments}
                  </p>
                </div>
              )}
            </div>
          </>
        )}
      </div>
    </div>
  );
}